#ifndef __CMidiEvent__
#define __CMidiEvent__

//	===========================================================================

#ifdef WIN32
	#pragma warning(disable : 4522)	//'Exponent::Midi::CMidiEvent' : multiple assignment operators specified
#endif

//	===========================================================================

#include "CMidi.hpp"
#include <Basics/CCountedObject.hpp>

//	===========================================================================

using Exponent::Midi::CMidi;
using Exponent::Basics::CCountedObject;

//	===========================================================================

namespace Exponent
{
	namespace Midi
	{
		/**
		 * @class CMidiEvent CMidiEvent.hpp
		 * @brief Wrapper class around a midi event
		 *
		 * @date 29/08/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CMidiEvent.hpp,v 1.7 2007/02/08 21:08:09 paul Exp $
		 */
		class CMidiEvent : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construction
			 * @param status Midi status
			 * @param value Byte 1 of the midi info
			 * @param subValue Bytes 2 of the midi info
			 * @param channel The miid channel
			 * @param delta The time delta, in samples from the start of the block (ion case of a midi file this is from the start of the file), block can be taken to mean\n
			 * the start of the last relevant position to the audio process
			 */
			CMidiEvent(const long status = CMidi::CMIDI_MIDI_NONE, const long value = 0, const long subValue = 0, const long channel = 1, const long delta = 0);

			/**
			 * Copy construction
			 * @param event The event to copy
			 */
			CMidiEvent(const CMidiEvent &event);

			/**
			 * Destruction
			 */
			virtual ~CMidiEvent();

//	===========================================================================

			/**
			 * Comparison for a qsort
			 * @param event1 The first event
			 * @param event2 The second event
			 * @retval <0 class1 goes before class2 0 class1 == class2 >0 class1 goes after class2
			 * @see CClassManager::classCompareFunction
			 * @see TPointerCollection::qsortCompare
			 */
			static int compareMidiEvents(const CMidiEvent **event1, const CMidiEvent **event2);

			/**
			 * Less than operator
			 * @param event The event to compare to
			 * @retval bool True if timeDelta is less than events time delta, false otherwise
			 */
			bool operator < (const CMidiEvent &event);

			/**
			 * Greater than operator
			 * @param event The event to compare to
			 * @retval bool True if timeDelta is greater than events time delta, false otherwise
			 */
			bool operator > (const CMidiEvent &event);

			/**
			 * Less than or equal operator
			 * @param event The event to compare to
			 * @retval bool True if timeDelta is less than or equal events time delta, false otherwise
			 */
			bool operator <= (const CMidiEvent &event);

			/**
			 * Greater than or euql operator
			 * @param event The event to compare to
			 * @retval bool True if timeDelta is greater than events time delta, false otherwise
			 */
			bool operator >= (const CMidiEvent &event);

			/**
			 * Assignment operator
			 * @param event The event to copy
			 * @retval CMidiEvent& A reference to this
			 */
			CMidiEvent &operator = (const CMidiEvent &event);

			/**
			 * Equality operator
			 * @param event The event to compare to
			 * @retval bool True if events are synchronous, false otherwise
			 * @see isSynchronousWith
			 */
			virtual bool operator == (const CMidiEvent &event);

			/**
			 * Inequality operator
			 * @param event The event to compare to
			 * @retval bool True if events are not synchronous, false otherwise
			 * @see isSynchronousWith
			 */
			virtual bool operator != (const CMidiEvent &event);

//	===========================================================================

			/**
			 * Get the status
			 * @retval const long The status
			 */
			FORCEINLINE const long getStatus() const { return m_status; }

			/**
			 * Get the value
			 * @retval const long Main value
			 */
			FORCEINLINE const long getValue() const { return m_value; }

			/**
			 * Get the sub value
			 * @retval const long Sub value
			 */
			FORCEINLINE const long getSubValue() const { return m_subValue; }

			/**
			 * Get the channel
			 * @retval const long The midi channel
			 */
			FORCEINLINE const long getChannel() const { return m_channel; }

			/**
			 * Get the time delta
			 * @retval const long The time delta
			 */
			FORCEINLINE const long getTimeDelta() const { return m_timeDelta; }

			/**
			 * Get pitch bend value
			 * @retval double The pitch bend value as a value -1 -> 1, where 0 is no pitch bend
			 */
			FORCEINLINE double getPitchBend() const { return m_pitchBend; }

//	===========================================================================

			/**
			 * Set the status
			 * @param status The status
			 */
			FORCEINLINE void setStatus(const long status) { m_status = status; }

			/**
			 * Set the value
			 * @param value The main value
			 */
			FORCEINLINE void setValue(const long value) { m_value = value; }

			/**
			 * Set the subvalue
			 * @param value The sub value
			 */
			FORCEINLINE void setSubValue(const long value) { m_subValue = value; }

			/**
			 * Set the channel
			 * @param channel The midi channel
			 */
			FORCEINLINE void setChannel(const long channel) { m_channel = channel; }

			/**
			 * Set the time delta
			 * @param delta The time delta
			 */
			FORCEINLINE void setTimeDelta(const long delta) { m_timeDelta = delta; }

			/**
			 * Set the pitch bend
			 * @param pitchBend Should be in the range -1 -> +1 with 0 being no bend 
			 */
			FORCEINLINE void setPitchBend(const double pitchBend) { m_pitchBend = pitchBend; }

			/**
			 * Set all values
			 * @param status Midi status
			 * @param value Byte 1 of the midi info
			 * @param subValue Bytes 2 of the midi info
			 * @param channel The miid channel
			 * @param delta The time delta
			 */
			FORCEINLINE void setEvent(const long status, const long value, const long subValue, const long channel, const long delta)
			{
				m_status    = status;
				m_value     = value;
				m_subValue  = subValue;
				m_channel   = channel;
				m_timeDelta = delta;
			}

//	===========================================================================

			/**
			 * Do the events happen at the same time
			 * @param event The event to compare with
			 * @retval bool True if time deltas are equal, false otherwise
			 */
			FORCEINLINE bool isSynchronousWith(const CMidiEvent &event)
			{
				return (event.m_timeDelta == m_timeDelta);
			}

//	===========================================================================

			/**
			 * Is this a note on message
			 * @retval bool True if Note on, false otherwise
			 */
			bool isNoteOn() const { return (m_status == CMidi::CMIDI_NOTE_ON && m_subValue != 0); }

			/**
			 * Is this a note off message
			 * @retval bool True if Note off, false otherwise
			 */
			bool isNoteOff() const { return ((m_status == CMidi::CMIDI_NOTE_OFF) || (m_status == CMidi::CMIDI_NOTE_ON && m_subValue == 0)); }

			/**
			 * Is this an all notes off message
			 * @retval bool True if all notes off, false otherwise
			 */
			bool isAllNotesOff() const { return (m_status == CMidi::CMIDI_ALL_NOTES_OFF); }

			/**
			 * Is control change message
			 * @retval bool True if control change message, false otherwise
			 */
			bool isControlChange() const { return (m_status == CMidi::CMIDI_CONTROL_CHANGE); }

			/**
			 * Is a pitch bend message
			 * @retval bool True if this is a pitch bend message, false otherwise
			 */
			bool isPitchBend() const { return (m_status == CMidi::CMIDI_PITCH_BEND); }

//	===========================================================================

			/**
			 * Get a description of the object
			 * @param string On return is filled with the description
			 * @param size The size of the stirng
			 */
			virtual void getObjectDescription(char *string, const long size) const;

//	===========================================================================

		protected:

//	===========================================================================

			long m_status;				/**< Note on, Note off, control change, all notes off.... */
			long m_value;				/**< note, cc number */
			long m_subValue;			/**< velocity, value */
			long m_channel;				/**< midi channel */
			long m_timeDelta;			/**< time delta within buffer in samples */
			double m_pitchBend;			/**< the pitch bend amount */
		};
	}
}
#endif	// End of CEvent.hpp